"""Database models for the testing system."""
from __future__ import annotations

from datetime import datetime
from typing import Optional

from flask_login import UserMixin
from werkzeug.security import check_password_hash, generate_password_hash

from . import db, login_manager


class Role:
    ADMIN = "admin"
    USER = "user"


@login_manager.user_loader
def load_user(user_id: str) -> Optional["User"]:
    """Load a user for Flask-Login."""
    return User.query.get(int(user_id))


class User(db.Model, UserMixin):
    """User of the testing system."""

    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(64), unique=True, nullable=False)
    last_name = db.Column(db.String(128), nullable=True)
    first_name = db.Column(db.String(128), nullable=True)
    middle_name = db.Column(db.String(128), nullable=True)
    department = db.Column(db.String(128), nullable=True)
    password_hash = db.Column(db.String(256), nullable=False)
    role = db.Column(db.String(16), default=Role.USER, nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow, nullable=False)

    tests_created = db.relationship("Test", back_populates="creator", lazy="select")
    test_attempts = db.relationship("TestAttempt", back_populates="user", lazy="select")
    test_assignments = db.relationship(
        "TestAssignment",
        back_populates="user",
        cascade="all, delete-orphan",
        lazy="select",
    )

    def set_password(self, password: str) -> None:
        self.password_hash = generate_password_hash(password)

    def check_password(self, password: str) -> bool:
        return check_password_hash(self.password_hash, password)

    @property
    def is_admin(self) -> bool:
        return self.role == Role.ADMIN

    @property
    def full_name(self) -> str:
        parts = [self.last_name, self.first_name, self.middle_name]
        return " ".join(filter(None, parts)).strip()

    def __repr__(self) -> str:  # pragma: no cover - debug helper
        return f"<User {self.username}>"


class Test(db.Model):
    """Test created by an administrator."""

    id = db.Column(db.Integer, primary_key=True)
    title = db.Column(db.String(128), nullable=False)
    description = db.Column(db.Text, nullable=True)
    created_by = db.Column(db.Integer, db.ForeignKey("user.id"), nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow, nullable=False)
    is_active = db.Column(db.Boolean, default=True, nullable=False)
    shuffle_questions = db.Column(db.Boolean, default=False, nullable=False)
    shuffle_answers = db.Column(db.Boolean, default=False, nullable=False)
    time_limit_minutes = db.Column(db.Integer, nullable=True)

    creator = db.relationship("User", back_populates="tests_created")
    questions = db.relationship(
        "Question",
        back_populates="test",
        cascade="all, delete-orphan",
        lazy="joined",
    )
    grade_criteria = db.relationship(
        "GradeCriteria",
        back_populates="test",
        cascade="all, delete-orphan",
        order_by="GradeCriteria.max_errors",
    )
    attempts = db.relationship("TestAttempt", back_populates="test", lazy="select")
    assignments = db.relationship(
        "TestAssignment",
        back_populates="test",
        cascade="all, delete-orphan",
        lazy="select",
    )

    def grade_for_errors(self, errors: int) -> int:
        """Determine grade from error count using configured criteria."""
        for criteria in sorted(self.grade_criteria, key=lambda c: c.max_errors):
            if errors <= criteria.max_errors:
                return criteria.grade
        return 2

    def __repr__(self) -> str:  # pragma: no cover
        return f"<Test {self.title}>"


class Question(db.Model):
    """Question in a test."""

    id = db.Column(db.Integer, primary_key=True)
    test_id = db.Column(db.Integer, db.ForeignKey("test.id"), nullable=False)
    question_text = db.Column(db.Text, nullable=False)
    question_type = db.Column(
        db.Enum("single", "multiple", name="question_type"),
        default="single",
        nullable=False,
    )

    test = db.relationship("Test", back_populates="questions")
    answers = db.relationship(
        "Answer",
        back_populates="question",
        cascade="all, delete-orphan",
        lazy="joined",
    )

    def __repr__(self) -> str:  # pragma: no cover
        return f"<Question {self.id} test={self.test_id}>"


class Answer(db.Model):
    """Answer option for a question."""

    id = db.Column(db.Integer, primary_key=True)
    question_id = db.Column(db.Integer, db.ForeignKey("question.id"), nullable=False)
    answer_text = db.Column(db.Text, nullable=False)
    is_correct = db.Column(db.Boolean, default=False, nullable=False)

    question = db.relationship("Question", back_populates="answers")

    def __repr__(self) -> str:  # pragma: no cover
        return f"<Answer {self.id} question={self.question_id}>"


class GradeCriteria(db.Model):
    """Grading rules for a test based on the number of mistakes."""

    id = db.Column(db.Integer, primary_key=True)
    test_id = db.Column(db.Integer, db.ForeignKey("test.id"), nullable=False)
    grade = db.Column(db.Integer, nullable=False)
    max_errors = db.Column(db.Integer, nullable=False)

    test = db.relationship("Test", back_populates="grade_criteria")

    def __repr__(self) -> str:  # pragma: no cover
        return f"<GradeCriteria grade={self.grade} max_errors={self.max_errors}>"


class TestAttempt(db.Model):
    """Attempt of a test by a user."""

    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(db.Integer, db.ForeignKey("user.id"), nullable=False)
    test_id = db.Column(db.Integer, db.ForeignKey("test.id"), nullable=False)
    started_at = db.Column(db.DateTime, default=datetime.utcnow, nullable=False)
    completed_at = db.Column(db.DateTime, nullable=True)
    score = db.Column(db.Integer, nullable=True)
    total_questions = db.Column(db.Integer, default=0, nullable=False)
    correct_answers = db.Column(db.Integer, default=0, nullable=False)
    expires_at = db.Column(db.DateTime, nullable=True)
    question_order = db.Column(db.Text, nullable=True)
    answer_order = db.Column(db.Text, nullable=True)

    user = db.relationship("User", back_populates="test_attempts")
    test = db.relationship("Test", back_populates="attempts")
    answers = db.relationship(
        "UserAnswer",
        back_populates="attempt",
        cascade="all, delete-orphan",
        lazy="dynamic",
    )

    def mark_completed(self, score: int, correct_answers: int, total_questions: int) -> None:
        self.completed_at = datetime.utcnow()
        self.score = score
        self.correct_answers = correct_answers
        self.total_questions = total_questions

    def __repr__(self) -> str:  # pragma: no cover
        return f"<TestAttempt user={self.user_id} test={self.test_id}>"


class UserAnswer(db.Model):
    """Answer choice made by a user within an attempt."""

    id = db.Column(db.Integer, primary_key=True)
    attempt_id = db.Column(db.Integer, db.ForeignKey("test_attempt.id"), nullable=False)
    question_id = db.Column(db.Integer, db.ForeignKey("question.id"), nullable=False)
    selected_answer_id = db.Column(db.Integer, db.ForeignKey("answer.id"), nullable=True)
    is_correct = db.Column(db.Boolean, default=False, nullable=False)

    attempt = db.relationship("TestAttempt", back_populates="answers")
    question = db.relationship("Question")
    selected_answer = db.relationship("Answer")

    def __repr__(self) -> str:  # pragma: no cover
        return f"<UserAnswer attempt={self.attempt_id} question={self.question_id}>"


class TestAssignment(db.Model):
    """Assignment of a test to a user or all users."""

    id = db.Column(db.Integer, primary_key=True)
    test_id = db.Column(db.Integer, db.ForeignKey("test.id"), nullable=False)
    user_id = db.Column(db.Integer, db.ForeignKey("user.id"), nullable=True)
    assigned_to_all = db.Column(db.Boolean, default=False, nullable=False)
    assigned_at = db.Column(db.DateTime, default=datetime.utcnow, nullable=False)

    test = db.relationship("Test", back_populates="assignments")
    user = db.relationship("User", back_populates="test_assignments")

    def __repr__(self) -> str:  # pragma: no cover
        return f"<TestAssignment test={self.test_id} user={self.user_id}>"
